; 10-Apr-2016 Fit triple star. read *in2 file, produce *.ou2 result
;==============================
; Translation of Kepler routine from ORBIT to IDL
; A. Tokovinin, Dec 27, 2004
; keyword /rho to get results in the form (theta,rho)
; /rv to get RV [primary, secondary]
;----------------------------------------
function eph, el, t, rho=rho
; Input: Orbital elements [PTEaWwi], time vector
; Output:  res = arr(t,xy) of relative coordinates

; Initialize the variables
  n = n_elements(t)
  res = fltarr(n,2) 
   pi2 = 2.*!dpi
   gr = 180D0/!dpi   
   P = EL[0]
   SF = EL[2]
   CF2 = 1.-SF*SF
   CF = SQRT(CF2)
   CF3 = CF*CF2
   EC = SQRT((1.+SF)/(1.-SF))
   CWW = COS(EL[4]/GR)
   SWW = SIN(EL[4]/GR)
   W = EL[5]/GR
   CW = COS( EL[5]/GR)
   SW = SIN( EL[5]/GR)
   SI = SIN(EL[6]/GR)
   CI = COS(EL[6]/GR)
;   TI = SI/CI

if keyword_set(rv) then begin 
    K1 = EL[7]
    K2 = EL[8]
    V0 = EL[9]
end else begin
; Thiele-van-den-Bos elements
        AA = EL[3]*(CW*CWW-SW*SWW*CI)
        BB = EL[3]*(CW*SWW+SW*CWW*CI)
        FF = EL[3]*(-SW*CWW-CW*SWW*CI)
        GG = EL[3]*(-SW*SWW+CW*CWW*CI)
endelse

for i=0,n-1 do begin
; Solve Kepler equation
        DT = T[i]-el[1]
        PHASE = dt/el[0] mod 1D0
        IF (PHASE lt 0.) THEN PHASE = PHASE+1.D0

        ANM = PHASE*2.*!dpi
        E = ANM

       E1 = E+(ANM+SF*SIN(E)-E)/(1.0-SF*COS(E))
       while (ABS(E1-E) gt 1.E-5) do begin
          E = E1
          E1 = E+(ANM+SF*SIN(E)-E)/(1.0-SF*COS(E))
        endwhile
        V = 2.*ATAN(EC*TAN(E1/2.))

        CV = COS(V)
        R = CF2/(1.+SF*CV)
        X = R*CV
        Y = R*SIN(V)
        RES(i,0) = AA*X+FF*Y
        RES(i,1) = BB*X+GG*Y
endfor

if keyword_set(rho) then begin ; theta/rho calculation
    rho = sqrt((res[*,0]^2) + res[*,1]^2 )
    theta = 180./!pi*atan(res[*,1], res[*,0])
    theta = (theta + 360.) mod 360
    res[*,0] = theta & res[*,1] = rho
endif     


return, res

end
; -------------- END ----------------
;
function eph2, el, t, rho=rho, rv=rv
; Input: Orbital elements [PTEaWwiK1K1V0], of outer and inner
; binaries, time vector
; Output:  res = arr(t,xy) of relative coordinates

res = eph(el[0:6],t) + eph(el[7:13],t)


if keyword_set(rho) then begin ; theta/rho calculation
    rho = sqrt((res[*,0]^2) + res[*,1]^2 )
    theta = 180./!pi*atan(res[*,1], res[*,0])
    theta = (theta + 360.) mod 360
    res[*,0] = theta & res[*,1] = rho
endif     

return, res

end
; -------------- END ----------------
; Read *.INP file
function getcoord, s  ; read decimal coordinate into HEX  
  l = strpos(s, '.')
  deg = fix(strmid(s,0,l))
  min = fix(strmid(s,l+1,2))
  sec = fix(strmid(s,l+3))
  res = abs(deg) + min/60. + sec/3600.
  if (deg lt 0) then res = -res
;  stop
  return, res
end
;------------------------------------------
pro correct, data, t0  ; Julian days or years?
 time = data[*,0] 
 for i=0,n_elements(time)-1 do begin

; years to JD-240000
  if (time[i] lt 3e3) and (t0 gt 3e3) then data[i,0] = 365.242198781D0 *(time[i]-1900.D0)+15020.31352D0

; JD-240000 to Besselian years
if (time[i] gt 3e3) and (t0 lt 3e3) then data[i,0] = 1900.0 + (time[i] - 15020.31352D0)/365.242198781D0  
 endfor 
end
;------------------------------------------------------------
; returns 3-element vector [deg,min,sec] from argument in degrees
     FUNCTION sixty,scalar
      ss=abs(3600.0d0*scalar)
      mm=abs(60.0d0*scalar) 
      dd=abs(scalar) 
      result=fltarr(3)
      result[0]=float(fix(dd))
      result[1]=float(fix(mm-60.0d0*result[0]))
      result[2]=float(ss-3600.d0*result[0]-60.0d0*result[1])
      if scalar[0] lt 0.0d0 then begin 
         if result[0] ne 0 then result[0] = -result[0] else $
         if result[1] ne 0 then result[1] = -result[1] else $
         result[2] = -result[2]
      endif

      return,result
      end
;------------------------------------------
pro readinp, fname


common xorb, xbase,obj, el,elerr, fixel, elname, pos,pos2,rv1,rv2, graph, editel, x,y,x2,y2,cov

el = dblarr(15)
elerr = fltarr(15)
fixel = replicate(1, 15) ; 1 if element is fitted

nmax = 1200 ; max number of observations 
pos = dblarr(nmax,7) ; time, PA, theta, err, O-C, O-C, PAorig
pos2 = dblarr(nmax,7) ; time, PA, theta, err, O-C, O-C, PAorig Inner subsystem resolved
;rv1 = dblarr(nmax,3)  ; time, Va, eVarv2 = dblarr(nmax,3)  ; time, Vb, eVb
;rv2 = dblarr(nmax,3)  ; time, Vb, eVarv2 = dblarr(nmax,3)  ; time, Vb, eVb


; WDS added
obj = {name:'', radeg:0D0, dedeg:0D0, wds:'00000+0000',npos:0,npos2:0, nrv1:0,nrv2:0, rms:fltarr(4), chi2n:fltarr(4), chi2:0., fname:fname}


 res = findfile(fname, count=c)
  if (c eq 0) then begin
    print, 'File ',fname,' is not found, exiting'
    obj.fname=''
    return
  endif

close, /all
openr, 11, fname

a=''
for i=0,19 do begin 
  readf, 11, a

  if strmid(a,0,1) eq '*' then begin 
    fix =0
    a= strmid(a,1)
  endif else fix=1
  if strmid(a,0,1) eq 'C' then continue

  t = strsplit(a, /extract)
;  print, '<'+t[0]+'>'
  ind = -1  ; index into element array

  case t[0] of
  'Object:': obj.name = strtrim(strmid(a,8))
  'RA:': obj.radeg = 15.D0*getcoord(t[1])
  'Dec:': obj.dedeg = getcoord(t[1])
  'WDS': obj.wds = t[1]
  'P': ind=0
  'T': ind=1
  'e': ind=2
  'a': ind=3
  'W': ind=4
  'w': ind=5
  'i': ind=6
  'P2': ind=0+7
  'T2': ind=1+7
  'e2': ind=2+7
  'a2': ind=3+7
  'W2': ind=4+7
  'w2': ind=5+7
  'i2': ind=6+7
  'B': ind=6+8
 
  else: print, 'Unknown tag  <'+t[0]+'>'
  endcase
if ind ge 0 then begin 
  el[ind] = double(t[1])
  fixel[ind] = fix
endif  
endfor

kpos=0
kpos2=0
krv1=0 & krv2=0
while not eof(11) do begin
  readf, 11, a
  if strmid(a,0,1) eq 'C' then continue

  t = strsplit(a, /extract)
  for k=0, n_elements(t)-1 do begin ; search ieach item for key
    if (strmid(t[k],0,2) eq 'I1') then begin ; position measure
      pos[kpos,0:3] = double(t[0:3])      
;     print, t[0], t[1], t[2]
     kpos++
     continue  ; next line
    endif
   if (strmid(t[k],0,2) eq 'I2') then begin ; position measure
      pos2[kpos2,0:3] = double(t[0:3])      
;     print, t[0], t[1], t[2]
     kpos2++
     continue  ; next line
    endif
  endfor
  nextline:  
endwhile

if kpos gt 0 then pos = pos[0:kpos-1,*] else pos=0
if kpos2 gt 0 then pos2 = pos2[0:kpos2-1,*] else pos2=0


correct, pos, el[1] 
correct, pos2, el[1] 


; Precession in THETA, degrees per year:
if (obj.radeg gt 0) and (kpos gt 0) and (el[1] lt 3000) then begin
   print, 'Correcting angles for precession' 
   PR=0.0057*SIN(obj.radeg/!radeg)/COS(obj.dedeg/!radeg) 
   pos[*,6] = pos[*,1] ; save original angles
   pos[*,1] += (2000.0 - pos[*,0])*PR ; corrected angles

   pos2[*,6] = pos2[*,1] ; save original angles
   pos2[*,1] += (2000.0 - pos2[*,0])*PR ; corrected angles


endif 

if (kpos gt 0) then begin 
  pos[*,4] = pos[*,2]*cos(pos[*,1]/!radeg) ; X points North
  pos[*,5] = pos[*,2]*sin(pos[*,1]/!radeg) ; Y points East
endif

if (kpos2 gt 0) then begin 
  pos2[*,4] = pos2[*,2]*cos(pos2[*,1]/!radeg) ; X points North
  pos2[*,5] = pos2[*,2]*sin(pos2[*,1]/!radeg) ; Y points East
endif



print, 'Position measures: ', kpos
print, 'Inner resolved measures: ', kpos2

obj.npos = kpos & obj.npos2 = kpos2 & obj.nrv1 = 0 & obj.nrv2 = 0

if (kpos gt 0) then if (max(pos[*,3] eq 0)) then begin
  print, 'Warning: zero errors encountered in input file! Stopping.' 
  stop
endif


graph.mode = 0

;graph.mode = 0 ; plot orbit by default 
;if (krv1 eq 0 ) then graph.mode=0 ; plot visual orbit only
;if (kpos eq 0 ) then graph.mode=1 ; plot SB orbit only


close, 11
end
;--------------------------------------
; Plot outer or inner orbit in XORB
; ps keyword to make a postscript file
; speckle=0.005 to plot all accurate measures with filled symbols

pro orbplot, ps=ps, speckle=speckle

common xorb

wsize = 600 ; 600 points in graphic window
margin=60   ; margin, points 
npts = 400 ; points in the curve
fact = 1.3 ; scale factor

name = strmid(obj.fname, 0, strpos(obj.fname, '.')) ; for plots
  gr = 180./!pi

if ((graph.mode eq 0) and (obj.npos gt 0)) then begin ; orbit plot

;  time = findgen(npts)/(npts-1)*el[0] + el[1]
  t1 = pos[0,0] 
;  time = findgen(npts)/(npts-1)*el[0] + t1
  time = findgen(npts)/(npts-1)*el[0]/2. + t1

  time2 = findgen(npts)/(npts-1)*el[7] + t1
  xye2 = eph(el[7:13],time2)/0.3 ; inner orbit

  xye = eph2(el, time) ; [*,2] array of orbit curve

  xobs = -pos[*,2]*sin(pos[*,1]/gr) ; data points
  yobs = pos[*,2]*cos(pos[*,1]/gr)

; find the plot scale
  tmp = [-xye[*,1], xobs] & xr = [min(tmp), max(tmp)]
  tmp = [xye[*,0], yobs] & yr = [min(tmp), max(tmp)]

  range = max([xr[1]-xr[0], yr[1]-yr[0]])*fact ; maximum range in both coordinates
  range = float(range)
  scale = (wsize-margin)/range ; from arseconds to points
  xcent = float(mean(xr)) & ycent = float(mean(yr))

; transform to pixel coordinates
  x = (xobs-xcent)*scale + wsize/2 + margin/2
  y = (yobs-ycent)*scale + wsize/2 + margin/2

  xy0 = eph2(el, pos[*,0])

if keyword_set(speckle) then begin 
  good = where(pos[*,3] le speckle, nsp) 
  bad =  where(pos[*,3] gt speckle, nbad) 
  print, 'Plotting inaccurate data as crosses: Nbad=',nbad
endif else  begin 
   good = findgen(obj.npos)
   nbad=0
endelse

  pp = float(margin)/wsize ; left margin
  pp1 = 0.995 ; right margin

if keyword_set(ps) then begin ; PS plot
   set_plot, 'ps'
   device, fi=name+'_POS1.ps', xs=14, ys=14
 endif

rx = range*[-0.5,0.5]+xcent
ry = range*[-0.5,0.5]+ycent

  plot, -xye[*,1], xye[*,0],/iso, li=0, xr=rx,yr=ry, xs=1, ys=1, position=[pp,pp, pp1,pp1]
  oplot, xobs[good], yobs[good], psym=6, symsize=0.3
  for i=0,n_elements(x)-1 do oplot, [xobs[i], -xy0[i,1]], [yobs[i],xy0[i,0]], li=1 
; oplot, [0,0], psym=2, symsize=2
 oplot, [0.], psym=2, symsize=2
  if (nbad gt 0) then oplot,  xobs[bad], yobs[bad], psym=7, symsize=0.5
; Inner pair
  oplot, -xye2[*,1], xye2[*,0], li=0
  oplot, -pos2[*,5], pos2[*,4], psym=5, symsize=0.3

   
if keyword_set(ps) then begin ; PS plot end
  device, /close
  print, 'Plot '+name+'_POS1.ps is produced'
  set_plot, 'x'
endif 

; stop

;--------------- inner orbit --------------------
endif else begin                 ; inner orbit

if (obj.npos2 eq 0) then begin
  print, 'No resolved measures of subsystem, returning'
  return
endif 

  t1 = pos2[0,0] 
  time = findgen(npts)/(npts-1)*el[7] + t1
  xye = eph(el[7:13], time)*el[14] ; [*,2] array of orbit curve

;stop
  xobs = -pos2[*,2]*sin(pos2[*,1]/gr) ; data points
  yobs = pos2[*,2]*cos(pos2[*,1]/gr)

; find the plot scale
  tmp = [-xye[*,1], xobs] & xr = [min(tmp), max(tmp)]
  tmp = [xye[*,0], yobs] & yr = [min(tmp), max(tmp)]

  range = max([xr[1]-xr[0], yr[1]-yr[0]])*fact ; maximum range in both coordinates
  range = float(range)
  scale = (wsize-margin)/range ; from arseconds to points
  xcent = float(mean(xr)) & ycent = float(mean(yr))

; transform to pixel coordinates
  x = (xobs-xcent)*scale + wsize/2 + margin/2
  y = (yobs-ycent)*scale + wsize/2 + margin/2
 
  xy0 = eph(el[7:13], pos2[*,0])*el[14]

if keyword_set(speckle) then begin 
  good = where(pos2[*,3] le speckle, nsp) 
  bad =  where(pos2[*,3] gt speckle, nbad) 
  print, 'Plotting inaccurate data as crosses: Nbad=',nbad
endif else  begin 
   good = findgen(obj.npos2)
   nbad=0
endelse


  pp = float(margin)/wsize ; left margin
  pp1 = 0.995 ; right margin

if keyword_set(ps) then begin ; PS plot
   set_plot, 'ps'
   device, fi=name+'_POS2.ps', xs=14, ys=14
endif

rx = range*[-0.5,0.5]+xcent
ry = range*[-0.5,0.5]+ycent

  plot, -xye[*,1], xye[*,0],/iso, li=0, xr=rx,yr=ry, xs=1, ys=1, position=[pp,pp, pp1,pp1]
  oplot, xobs[good], yobs[good], psym=6, symsize=0.5
  for i=0,n_elements(x)-1 do oplot, [xobs[i], -xy0[i,1]], [yobs[i],xy0[i,0]], li=1 
; oplot, [0,0], psym=2, symsize=2
 oplot, [0.], psym=2, symsize=2
  if (nbad gt 0) then oplot,  xobs[bad], yobs[bad], psym=7, symsize=0.5
  
 
if keyword_set(ps) then begin ; PS plot end
  device, /close
  print, 'Plot '+name+'_POS2.ps is produced'
  set_plot, 'x'
endif 


endelse ; inner orbit

;stop
; return to caller
end

;----------------------------------------------------------------
; Fit orbital elements by Levenberg-Marquardt method
;----------------------------------------------------------
; calculates ephemeride and derivatives in theta, rho, RV1, RV1
; num is the number of data 'point', total 2*npos+nrv
; only fitted orbital elements are in par
function alleph, i, par

common xorb


e = 0.01 ; relative element variation for derivative calculation
del = [e*el[0], e*el[0], e, e*el[3], 1., 1., 1., $
       e*el[7], e*el[7], e, e*el[10], 1., 1., 1.,e]


selfit = where(fixel gt 0, nfit) ; selection of fitted elements
el0 = el
el0[selfit] = par ; temporary elements

result = fltarr(15) ; all derivatives

if (i lt 2*obj.npos) then begin ; XY of outer pair
  if (i ge obj.npos) then j=1 else j=0 ; j=1 for Y, j=0 for X
  time = pos[i - j*obj.npos,0]
  res = (eph2(el0,[time]))[j] ; X or Y, depending on jj
  for k=0,14 do if (fixel[k] gt 0) then begin ; cal. derivative
    el1 = el0 
    el1[k] += del[k]
    delta = (eph2(el1,[time]))[j] - res
    result[k] = delta/del[k] 
   endif ; fixel in for loop
;stop

endif else begin ; resolved measure
 if (i ge 2*obj.npos+obj.npos2) then j=1 else j=0 ; j=1 for X, j=0 for Y
  time = pos2[i - 2*obj.npos -j*obj.npos2,0]
  res = (eph(el0[7:13],[time]))[j]*el0[14] ; X or Y, depending on j
  for k=7,14 do if (fixel[k] gt 0) then begin ; cal. derivative
    el1 = el0 
    el1[k] += del[k]
    delta = (eph(el1[7:13],[time]))[j]*el1[14] - res
    result[k] = delta/del[k] 

; debug
;  if (j eq 1) then begin  
;     print, 'i, Time, Eph [0,dx]:]', time, res,  (eph(el1[7:13],[time]))[j]*el1[14]; 
;     print, result[k], del[k]
;     stop
;   endif

 endif ; fixel[k]
 
endelse ; resolved mesure of subsystem

;  result[0] = res
  result = [res,result[selfit]]

;  stop  
  return, result
end
;----------------------------------------------------------


;----------------------------------------------------------
; keyword rms to calculate RMS only
pro fitorb, rms=rms

common xorb

;common xorb, xbase,obj, el,elerr, fixel, elname, pos, rv1,rv2, graph, editel, x,y,x2,y2,cov


; Calculate the observables
npos = obj.npos & npos2=obj.npos2 &  nrv1 = 0 & nrv2=0

n = 2*npos + 2*npos2 ; total number or points
yy = fltarr(n); X, Y, RV1, RV2
err = fltarr(n) ; measure errors

if (npos gt 0) then begin 
  yy[0:npos-1] = pos[*,4]      ; X-position
  err[0:npos-1] = pos[*,3] ; error in X
  yy[npos:2*npos-1] = pos[*,5] ; Y-position
  err[npos:2*npos-1] =  pos[*,3] ; error in Y
endif

if (npos2 gt 0) then begin 
  yy[npos*2:npos*2+npos2-1] = pos2[*,4]      ; X-position
  err[npos*2:npos*2+npos2-1] = pos2[*,3] ; error in X
  yy[npos*2+npos2: npos*2+2*npos2-1] = pos2[*,5] ; Y-position
  err[npos*2+npos2:npos*2+2*npos2-1] =  pos2[*,3] ; error in Y
endif

; fita = fixel ; zero if fixed
selfit = where(fixel gt 0, nfit)
print, 'Fitting ', nfit, ' elements'
par = el[selfit]


ix = findgen(n) ; fictitious argument
y1 = fltarr(n) ; ephemeris
for i=0,n-1 do y1[i] = (alleph(i, par))[0] ; ephemeris

;plot, y[0:npos-1]
;oplot, y1[0:npos-1], psym=1

; residual calculation
nmin = [0,npos,2*npos,2*npos+npos2] ; left limit
nmax = [npos, 2*npos, 2*npos+npos2, 2*npos+2*npos2]
ndat = nmax-nmin ; data points of each type

wt = err^(-2.) ; weight
resid2 = (yy-y1)^2*wt

sd= fltarr(4) ; res^2/sig2
wsum = fltarr(4)
normchi2 = fltarr(4)

for j=0,3 do if ndat[j] gt 0 then begin 
 sd[j]= total(resid2[nmin[j]:nmax[j]-1])
 wsum[j] = total(wt[nmin[j]:nmax[j]-1])
 normchi2[j] = sd[j]/ndat[j]
endif 
; weighted rms
wrms = fltarr(4)
wsel = where(wsum gt 0)
wrms[wsel] = sqrt(sd[wsel]/wsum[wsel]) ; weighted rms

print, 'CHI2/N: ', normchi2, format='(A,4F8.2)'
print, 'RMS in Theta,rho,Theta2,rho2: ', wrms, format='(A,4F8.4)'

;stop

if keyword_set(rms) then begin ; only RMS is needed, no fit
  obj.rms = wrms
  obj.chi2n = normchi2
  return
endif

;y1 = lmfit(ix, yy, par, chisq=chi2, fita=fita, measure_errors=err, function_name='alleph',sigma=sigma, iter=iter, itmax=30) 

;y1 = lmfit(ix, yy, par, chisq=chi2, measure_errors=err, function_name='alleph',sigma=sigma, iter=iter, itmax=30, cov=cov) 

y1 = lmfit(ix, yy, par, chisq=chi2, measure_errors=err, function_name='alleph',sigma=sigma, iter=iter, itmax=30, cov=cov, convergence=convergence, tol=1e-10) 


; sqrt(cov[i,i]) equals sigma[i]
print, 'LM iterations, convergence: ',iter, convergence
print, 'CHI2,M=', chi2, 2*npos+2*npos2-nfit, format='(A,F8.2,I6)'

; stop

;print, 'New elements:', par

; recalculate residuals
resid2 = (yy-y1)^2*wt
for j=0,3 do if ndat[j] gt 0 then begin 
 sd[j]= total(resid2[nmin[j]:nmax[j]-1])
 wsum[j] = total(wt[nmin[j]:nmax[j]-1])
 normchi2[j] = sd[j]/ndat[j]
endif 
; weighted rms
wrms = fltarr(4)
wsel = where(wsum gt 0)
wrms[wsel] = sqrt(sd[wsel]/wsum[wsel]) ; weighted rms

print, 'CHI2/N: ', normchi2, format='(A,4F8.2)'
print, 'RMS in X1,Y1,X2,Y2: ', wrms, format='(A,4F8.4)'

; stop

  obj.rms = wrms
  obj.chi2n = normchi2
  obj.chi2 = chi2

;stop


el[selfit] = par       ; update elements
elerr[selfit] = sigma  ; element errors


showdat
orbplot

;stop
end
;----------------------------------------------------------
;----------------------------------------------------------
;----------------------------------------------------------
 ; Save file
pro orbsave

common xorb


fname = strmid(obj.fname, 0, strpos(obj.fname, '.'))+'.ou2'

ra1 = sixty(obj.radeg/15.D0) ; 3-element vector [hh, mm, ss]
dec1 = abs(sixty(obj.dedeg))

ra = string(ra1[0]+0.01*ra1[1]+1e-4*ra1[2], '(F9.6)')
dec = string(dec1[0]+0.01*dec1[1]+1e-4*dec1[2], '(F09.6)')
if obj.dedeg lt 0 then dec = '-'+ dec


elfmt = ['F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2', $
'F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2','F8.3']

elname=['P','T','e','a','W','w','i','P2','T2','e2','a2','W2','w2','i2','B']

 ; fixed elements 

;stop

close, /all
openw, 11, fname
printf, 11, 'Object: '+obj.name
printf, 11, 'RA:     '+ra
printf, 11, 'Dec:    '+dec
printf, 11, 'WDS:    '+obj.wds
if (obj.chi2 eq 0) then fitorb, /rms ; calculate residuals

; print orbital elements
for i=0,14 do printf, 11, elname[i], el[i], elerr[i], format='(A-2, 6X, '+elfmt[i]+', 8X, '+elfmt[i]+')'

; print CHI2 and rms
printf, 11, 'C','RMS:','X','Y', 'X2','Y2', format='(A1,5A10)'
printf, 11, 'C', obj.rms, format='(A1,10X, F10.2,F10.4,2F10.3)'
printf, 11, 'C','CHI2', 'X','Y', 'X2','Y2', format='(A1,5A10)'
printf, 11, 'C',obj.chi2, obj.chi2n,  format='(A1,5F10.2)'
; Print period and T0 in years
if (el[1] gt 10000.) then  begin 
   tmp = [el[0]/365.2421987, 1900. +  (el[1] - 15020.31352D0)/365.242198781D0] 
   tmp1 = elerr[0:1]/365.2421987
endif else begin 
   tmp = [el[0]*365.24219878, 365.242198781D0 *(el[1]-1900.D0)+15020.31352D0]
   tmp1 = elerr[0:1]*365.2421987
endelse
  printf, 11, 'C P,T= ', tmp, format='(A, F12.5,3X,F10.4)'
  print,  'P,T= ', tmp, format='(A, F12.5,3X,F10.4)'
  printf, 11, ' +- ', tmp1, format='(A, F12.5,3X,F10.4)'
  print,  '+-  ', tmp1, format='(A, F12.5,3X,F10.4)'

; Position measures
if (obj.npos gt 0) then begin
 idx = [0,6,2,3] ; select right data from pos, that is uncorrected angles
 res = eph2(el,pos[*,0], /rho) ; theta, rho
 for i=0,obj.npos-1 do printf, 11, pos[i,idx], pos[i,1]-res[i,0],  pos[i,2]-res[i,1], 'I1', $
  format='(F10.4, F8.1,2F8.4,F8.1,F8.4,2X,A2)'
endif

; Inner-system measures measures
if (obj.npos2 gt 0) then begin
printf, 11, 'C Resolved measures: '
 idx = [0,6,2,3] ; select right data from pos, that is uncorrected angles
 res = eph(el[7:13],pos2[*,0], /rho) ; theta, rho
 res[*,1]*=el[14]; scale
 for i=0,obj.npos2-1 do printf, 11, pos2[i,idx], pos2[i,1]-res[i,0],  pos2[i,2]-res[i,1], 'I2', $
  format='(F10.4, F8.1,2F8.4,F8.1,F8.4,2X,A2)'
endif


;stop

  print, 'Results saved in '+fname
  close, 11
  return ; no RV observations


end
;---------------------------------------------
;---- GUI part of the program ----------------
;---------------------------------------------
; GUI for combined orbit fitting
; Based on ORBIT Fortran program by Tokovinin, 1995-1999
pro xorb_event, ev

common xorb
; xbase,obj, el,elerr, fixel, elname, pos, rv1,rv2, graph, editel, x,y,x2,y2


 type = tag_names(ev, /structure) ; name of the {ev} structure
; print, 'EV type is: ',type


  case type of
;--------------------
'WIDGET_BUTTON': begin   
 widget_control,ev.id, get_value=value 
 case value of
  'Exit': widget_control, ev.top, /destroy
  'Fit': fitorb  
  'Save': orbsave
  'POS': begin
    graph.mode = 0
    orbplot
    print, 'Use orbplot, /ps, speckle=0.02 to make the orbit file'
  end 
  'POS2': begin
    graph.mode = 1
    orbplot
  end 
 'Open': begin
        bas0 = widget_base(title='File selector',/column)      
        filesel = cw_filesel(bas0) 
        widget_control, bas0, /realize
        xmanager, 'xorb', bas0
    end
  'Reopen': begin 
    readinp, obj.fname
    showdat
    orbplot
  end
  else:
endcase ; button
; Fix-element buttons
 for i=0,14 do if (ev.id eq editel[i,1]) then begin 
;  print, 'Fix element ',i
  fixel[i] = (1 + fixel[i]) mod 2
  widget_control, editel[i,1], set_value=string(fixel[i],'(I1)')

 endif
 end

'WIDGET_DRAW': begin
;  print, 'Draw event type ',  ev.type
  if (ev.type eq 0) then begin
    xc = ev.x &  yc = ev.y    
;    print, 'X,Y: ', xc, yc

    dist = (x-xc)^2 + (y-yc)^2
    sel = (where(dist eq min(dist)))[0]

if (graph.mode eq 0) then   print, pos[sel,0:3], format='(F12.4, F8.2, 2F8.3)' else $
      print, pos2[sel,0:3], format='(F12.4, F8.2, 2F8.3)' 


  endif ; draw widget

 end ; draw events 
'WIDGET_DROPLIST': begin
;  print, 'Menu ', ev.index
 case ev.index of
 0: begin
   print, 'Ephemeride. Enter 0 parameter to'
   n=0
   read, prompt='Start time: ', tstart
   if tstart eq 0 then return 
   read, prompt='Number of points (0 to skip): ', n
   if n eq 0 then return 
  
   if (n gt 1) then begin 
     read, prompt='Time step: ', dt
     t = tstart + findgen(n)*dt
   endif else t = [tstart]
   t1 = t
 ; Julian days or years?
  if (t[0] lt 3e3) and (el[1] gt 3e3) then t1 = 365.242198781D0 *(t1-1900.D0)+15020.31352D0
  if (t[0] gt 3e3) and (el[1] lt 3e3) then t1 = 1900.0 + (t1 - 15020.31352D0)/365.242198781D0  

 ;   print, t
  res = eph2(el,t1, /rho)
  if (obj.nrv1 gt 0) then begin 
      res2 = eph(el,t1, /rv) 
      print, '   Time      PA      Sep      RV(A)      RV(B)' 
      for i=0,n-1 do print, t[i], res[i,*], res2[i,*], format='(F10.3, F8.2, F8.4, 2F8.2)'
   endif else begin 
     print, '   Time      PA      Sep' 
      for i=0,n-1 do print, t[i], res[i,*], format='(F10.3, F8.2, F8.4, 2F8.2)'
   endelse 
   end ; ephemeride
 1: begin
   print, 'Mass sum from parallax'
   read,  prompt='Parallax in the same units as axis (0 to skip): ', plx
   if plx eq 0 then return 
   msum, plx
   end
 2: begin
   print, 'Spectroscopic masses'
   msin3i
   end
 3: begin
   print, 'Min. secondary mass'
   read,  prompt='Mass of the primary (0 to skip): ', mass1
   if mass1 eq 0 then return 
   m2min, mass1
   end
 4: plotrho
   else:
   endcase
 end ; droplist
  ;--------------------
'FILESEL_EVENT': begin
case ev.done of
0: begin
   obj.fname=''
  end
1: begin
    widget_control,ev.id, get_value=value    

    slash = strpos(value, '/', /reverse_search)
    if (slash gt -1) then obj.fname = strmid(value,slash+1)
    print, 'Selected file: ', obj.fname
    readinp, obj.fname
    if (obj.fname eq '') then return ; not found  
    widget_control, ev.top, /destroy
    showdat
    orbplot
   end
2:  widget_control, ev.top, /destroy
  endcase
end ; filesel event

;--------------------------- Edit events
else: begin
 widget_control,ev.id, get_uvalue=uvalue 
 if (uvalue eq 'Paredit') then begin
    widget_control,ev.id, get_value=value 
    print, 'Edit event'
    
    for i=0,14 do if (ev.id eq editel[i,0]) then  begin
      el[i] = value
      print, 'Edit element '+elname[i] 
      continue
   endif

  endif
end ; edit events


endcase 
end
;------------------------------------------------------------
pro xorb, fname

common xorb



elfmt = ['F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2', $
'F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2','F8.3']

elname=['P','T','e','a','W','w','i','P2','T2','e2','a2','W2','w2','i2','B']


graph={mode:0,points:0L, curve:0L}
editel = lonarr(15,3) ; widget IDs for elements and fixes 

print, 'Calculation of visual & spectroscopic orbits. A.Tokovinin, 2013'
;print, 'Use commands msum, plx and plotrho'

if n_elements(fname) eq 0 then begin ; 
  print, "Correct call: XORB, 'Input-File-Name'"
  return
endif

readinp, fname
if (obj.fname eq '') then return ; not found

options=['Ephemeride', 'Mass sum', 'M sin3i', 'M2min', 'Plot separation']


  xbase = widget_base(title='ORBIT GUI',/column)  ; main base for GUI

;---- Upper row: File, data file
  base0 =  widget_base(xbase, /row)
  junk = widget_button(base0, xsize=60, value='Open',sensitive=1)
  junk = widget_button(base0, xsize=60, value='Reopen',sensitive=1)
  junk = widget_label(base0, xsize=150, uname='fname_label',  value='X')
  junk = widget_label(base0, xsize=150, uname='npos_label',  value='X')
  junk = widget_label(base0, xsize=150, uname='nrv_label',  value='X')


;---- 2nd row: graphical window and elements
   base1 =  widget_base(xbase, /row)

  draw = widget_draw(base1,xsize=600,ysize=600,/button_events) ; plot window

  widget_control, xbase, /realize
  widget_control, draw, get_value=ws0
  wset, ws0

  base2 =  widget_base(base1, /column)

  for i=0,6 do begin ; orbit elements
   junk = widget_base(base2, /row)
 
  editel[i,0] =  cw_field(junk,uname='El:'+elname[i],title=elname[i],uvalue='Paredit',xsize=16,/return_events,value='')
   editel[i,1] = widget_button(junk, xsize=20, uvalue='Fixel', value='',sensitive=1)
   editel[i,2] = widget_label(junk, xsize=60, uvalue='', value='X')

  editel[i+7,0] =  cw_field(junk,uname='El:'+elname[i+7],title=elname[i+7],uvalue='Paredit',xsize=16,/return_events,value='')
   editel[i+7,1] = widget_button(junk, xsize=30, uvalue='Fixel', value='',sensitive=1)
   editel[i+7,2] = widget_label(junk, xsize=100, uvalue='', value='X')
endfor
; Beta-line, i=14
 i=14
 junk = widget_base(base2, /row)
  editel[i,0] =  cw_field(junk,uname='El:'+elname[i],title=elname[i],uvalue='Paredit',xsize=16,/return_events,value='')
   editel[i,1] = widget_button(junk, xsize=30, uvalue='Fixel', value='',sensitive=1)
   editel[i,2] = widget_label(junk, xsize=100, uvalue='', value='X')


  junk = widget_label(base2, xsize=150, value='---------------------------------------')

   base2a =  widget_base(base2, /row)
  junk = widget_label(base2a, value='Analysis ')
  junk = widget_droplist(base2a,uname='more', uvalue='more', value=options) ; drop-down menu
;  junk = widget_button(base2, xsize=60, value='MORE',sensitive=0)
  junk = widget_button(base2, xsize=60, value='Fit',sensitive=1)
  junk = widget_button(base2, xsize=60, value='Save',sensitive=1)
  junk = widget_button(base2, xsize=60, value='Exit',sensitive=1)

; ----- Bottom row: graph plots 

  base3 =  widget_base(xbase, /row)
;  junk = widget_combobox(base3,uname='gr_mode',xsize=100,value=['POS','RV'])

  junk = widget_button(base3, xsize=60, value='POS',sensitive=1)
  junk = widget_button(base3, xsize=60, value='POS2',sensitive=1)

;  junk = widget_button(base3, xsize=60, value='Exit',sensitive=1)

 xmanager, 'xorb', xbase, /no_block

 showdat
 orbplot

end
;------------------------------------------------------------
; display orbital elements
pro showdat

common xorb

elfmt = ['F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2', $
'F12.5','F10.4','F6.4', 'F8.4', 'F8.2', 'F8.2', 'F8.2','F8.4']


  for i=0,14 do begin 
     widget_control, editel[i,0], set_value=string(el[i],format='('+elfmt[i]+')')
     widget_control, editel[i,1], set_value=string(fixel[i],'(I1)')
     widget_control, editel[i,2], set_value=string(elerr[i],format='('+elfmt[i]+')')
  endfor

 widget_control,  widget_info(xbase,find_by_uname='fname_label'), set_value=obj.fname
 widget_control,  widget_info(xbase,find_by_uname='npos_label'), set_value='Npos: '+string(obj.npos)
 widget_control,  widget_info(xbase,find_by_uname='nrv_label'), set_value='NRV: '+string(obj.nrv1)+'   '+string(obj.nrv2)


end
;------------------------------------------------------------
; print mass sum, input: parallax in arcsec
pro msum, plx, inner=inner

common xorb

if keyword_set(inner) then begin
  b = 1./el[14]
  q = b/(1. - b) 
  print, 'Inner subsystem, q= ',q 
  a = el[10]*el[14]/plx
  p = el[7]
endif else begin
  print, 'Outer system'
  a = el[3]/plx
  p = el[0]
endelse

if el[1] gt 10000 then p= p/365.2422 ; period in years 

print, 'Mass sum: ', a^3/p^2

end
;------------------------------------------------------------
pro plotrho
common xorb
time = pos[*,0]
xye = eph2(el, time) ; [*,2] 
rho = sqrt(total(xye^2,2))

plot, time, pos[*,2], psym=1
oplot, time, rho
end
;------------------------------------------------------------
pro plotx
common xorb
time = pos[*,0]
xye = eph2(el, time) ; [*,2] 

plot, time, pos[*,4], psym=1
oplot, time, xye[*,0]
end
;------------------------------------------------------------
pro ploty
common xorb
time = pos[*,0]
xye = eph2(el, time) ; [*,2] 

plot, time, pos[*,5], psym=1
oplot, time, xye[*,1]
end
;------------------------------------------------------------
pro covmat
common xorb

elname=['P','T','e','a','W','w','i','P2','T2','e2','a2','W2','w2','i2','B']

selfit = where(fixel gt 0, nfit)
print, 'Fitting ', nfit, ' elements'


print, elname[selfit], format='(8X,14A-8)' ; title line
for i=0,nfit-1 do begin
  rho = cov[i,0:i]/sqrt(cov[i,i]) 
  for j=0,i do rho[j] = rho[j]/sqrt(cov[j,j])
  print, elname[selfit[i]], rho, format='(A2, 14F8.2)'
endfor ;i
print, elname[selfit], format='(8X,14A-8)' ; title line

end
;------------------------------------------------------------
;------------------------------------------------------------
; Weight down outliers
pro outliers, nsigma

common xorb

; recalculate residuals

res = eph2(el,pos[*,0]) ; [*,XY]

res2x = ((pos[*,4] - res[*,0])/pos[*,3])^2
res2y = ((pos[*,5] - res[*,1])/pos[*,3])^2

chi2 = (total(res2x) + total(res2y))/(2.*obj.npos) ; mean
print, 'Mean CHI2/N = ', chi2

threshold = nsigma^2*chi2

res = res2x > res2y 

;bad = where((res2x gt threshold) or (res2y gt nsigma*chi2), nbad)
bad = where(res gt threshold, nbad)

if (nbad eq 0) then begin
  print, 'No outliers!'
  return
endif
print, nbad, ' outliers detected at times:'
print, pos[bad,0]
print, '.C to proceed with correction, RETALL to quit'
stop
; increase errors to get 1-sigma outliers
for j=0,nbad-1 do pos[bad[j],3] *= sqrt( res[bad[j]]/(chi2 > 1.))
print, 'DONE'
end
;------------------------------------------------------------

